%% SEIR parameter studies

% Runs an SEIR experiment to determine how maximum infectious fraction and 
% final at-risk fraction depend on a parameter value
%
% Uses seir_sim.m, 2020/10/27 version
%
% User specifies a list of values for one of the key parameters:
%   R0 is the basic reproductive number
%   TL is the mean incubation time
%   TI is the mean infectious duration
%   I0 is the initial infectious fraction
%   V is the initial immune fraction
%   delta is a contact factor that reduces infectivity
%
% The program is designed so that only a few lines need to be modified to
% make a new experiment (see '%%%' comments)
%   lines 56-58 define the independent variable values
%   line 61 identifies the x axis label for the graph
%   line 89 links the independent variable name and values
%
% Output figure:
%   top left panel: max infectious fraction
%   top right panel: day of max new infections
%   bottom left panel: final fraction susceptibles
%   bottom right panel: day when end condition is reached
%
% Y contains one row for each value of the independent variable
% Columns are
%   1. maximum infectious fraction
%   2. day of max new infections
%   3. final susceptible fraction
%   4. days until I<target
%
% by Glenn Ledder
% written 2020/08/15
% revised 2020/10/27
%
% direct comments to gledder@unl.edu

%% DEFAULT SCENARIO DATA

R0 = 2.5;
TL = 5;
TI = 10;
I0 = 0.001;
V = 0;
delta = 1;

%% INDEPENDENT VARIABLE DATA

%%% This section needs to be modified for each experiment.

%%% first and last are the min and max values for the independent variable
%%% N is the number of points (not subdivisions)
first = 0;
last = 5;
N = 51;

%%% xname is the name for the x axis label
xname = 'R0';

%% COMMON DATA

target = 0.001;

%% INITIALIZATION

opengl hardwarebasic;
clf
for k=1:4
    subplot(2,2,k)
    hold on
    xlabel(xname)
end

% xvals holds whatever values are being used for the independent variable
xvals = linspace(first,last,N);

% rows are for each n
Y = zeros(N,4);

%% COMPUTATION

for n=1:N
    
    %%% The left side of this statement needs to be the independent
    %%% variable for the experiment.
    R0 = xvals(n);

    eta = 1/TL;
    gamma = 1/TI;
    beta = gamma*R0;

    b = eta+gamma;
    c = eta*(gamma-beta*(1-V));
    lambda = 0.5*(-b+sqrt(b^2-4*c));
    rho = (lambda+gamma)/eta;
    E0 = rho*I0;

    [S,~,I,~] = seir_sim(delta*beta,eta,gamma,E0,I0,V,target);

    [Y(n,1),Y(n,2)] = max(I);
    Y(n,2) = Y(n,2)-1;

    Y(n,3) = S(end);
    Y(n,4) = length(I)-1;
end
    
%% OUTPUT

for k=1:4
    subplot(2,2,k)
    plot(xvals,Y(:,k),'LineWidth',1.7)
end

subplot(2,2,1)
ylabel('max fraction infectious')
subplot(2,2,2)
ylabel('days for max infectious')
subplot(2,2,3)
ylabel('final fraction susceptible')
subplot(2,2,4)
ylabel('days to end condition')
